<?php

namespace App\Controller\Admin;

use App\Entity\HomeCard;
use App\Form\HomeCardType;
use App\Repository\HomeCardRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/home/card')]
final class HomeCardController extends AbstractController
{
    #[Route(name: 'app_admin_home_card_index', methods: ['GET'])]
    public function index(HomeCardRepository $homeCardRepository): Response
    {
        return $this->render('admin/home_card/index.html.twig', [
            'home_cards' => $homeCardRepository->findBy([], [
                'rank' => 'ASC',
            ]),
            'maxRank' => $homeCardRepository->findMaxRank(),
        ]);
    }

    #[Route('/new', name: 'app_admin_home_card_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $homeCard = new HomeCard();
        $form = $this->createForm(HomeCardType::class, $homeCard);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($homeCard);
            $entityManager->flush();
            $this->addFlash('success', 'Le card a été ajouté avec succès');

            return $this->redirectToRoute('app_admin_home_card_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/home_card/new.html.twig', [
            'home_card' => $homeCard,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_admin_home_card_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, HomeCard $homeCard, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(HomeCardType::class, $homeCard);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();
            $this->addFlash('success', 'Le card a été ajouté avec succès');

            return $this->redirectToRoute('app_admin_home_card_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/home_card/edit.html.twig', [
            'home_card' => $homeCard,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_admin_home_card_delete', methods: ['POST'])]
    public function delete(Request $request, HomeCard $homeCard, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$homeCard->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($homeCard);
            $entityManager->flush();
            $this->addFlash('success', 'Le card a été supprimé avec succès');
        } else {
            $this->addFlash('success', 'Le card n\'a pas pu être supprimé');
        }

        return $this->redirectToRoute('app_admin_home_card_index', [], Response::HTTP_SEE_OTHER);
    }

    #[Route('/up/{id}', name: 'app_admin_home_card_move_up', methods: ['POST'])]
    public function moveUp(Request $request, HomeCard $homeCard, EntityManagerInterface $em): Response
    {
        if ($this->isCsrfTokenValid('move-up'.$homeCard->getId(), $request->getPayload()->getString('_token'))) {
            $homeCard->setRank($homeCard->getRank() + 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du card a été avancé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du card n\'a pas pu être avancé');
        }
        
        return $this->redirectToRoute('app_admin_home_card_index', [], Response::HTTP_SEE_OTHER);
    }
    
    #[Route('/down/{id}', name: 'app_admin_home_card_move_down', methods: ['POST'])]
    public function moveDown(Request $request, HomeCard $homeCard, EntityManagerInterface $em): Response
    {
        if ($homeCard->getRank() > 0 && $this->isCsrfTokenValid('move-down'.$homeCard->getId(), $request->getPayload()->getString('_token'))) {
            $homeCard->setRank($homeCard->getRank() - 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du card a été reculé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du card n\'a pas pu être reculé');
        }

        return $this->redirectToRoute('app_admin_home_card_index', [], Response::HTTP_SEE_OTHER);
    }
}
