<?php

namespace App\Controller\Admin;

use App\Entity\Submenu;
use App\Form\SubmenuType;
use App\Repository\SubmenuRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/submenu')]
final class SubmenuController extends AbstractController
{
    #[Route(name: 'app_admin_submenu_index', methods: ['GET'])]
    public function index(SubmenuRepository $submenuRepository): Response
    {

        return $this->render('admin/submenu/index.html.twig', [
            'submenus' => $submenuRepository->findBy([], [
                'menu' => 'ASC',
                'rank' => 'ASC',
            ]),
            'maxRank' => array_column($submenuRepository->findMaxRank(), 'maxRank', 'menuId'),
        ]);
    }

    #[Route('/new', name: 'app_admin_submenu_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $submenu = new Submenu();
        $form = $this->createForm(SubmenuType::class, $submenu);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($submenu);
            $entityManager->flush();
            $this->addFlash('success', 'Le sous-menu a été ajouté avec succès');

            return $this->redirectToRoute('app_admin_submenu_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/submenu/new.html.twig', [
            'submenu' => $submenu,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_admin_submenu_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Submenu $submenu, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(SubmenuType::class, $submenu);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();
            $this->addFlash('success', 'Le sous-menu a été modifié avec succès');

            return $this->redirectToRoute('app_admin_submenu_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/submenu/edit.html.twig', [
            'submenu' => $submenu,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_admin_submenu_delete', methods: ['POST'])]
    public function delete(Request $request, Submenu $submenu, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$submenu->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($submenu);
            $entityManager->flush();
            $this->addFlash('success', 'Le sous-menu a été supprimé avec succès');
        } else {
            $this->addFlash('error', 'Le sous-menu n\'a pas pu être supprimée ');
        }

        return $this->redirectToRoute('app_admin_submenu_index', [], Response::HTTP_SEE_OTHER);
    }

    #[Route('/up/{id}', name: 'app_admin_submenu_move_up', methods: ['POST'])]
    public function moveUp(Request $request, Submenu $submenu, EntityManagerInterface $em): Response
    {
        if ($this->isCsrfTokenValid('move-up'.$submenu->getId(), $request->getPayload()->getString('_token'))) {
            $submenu->setRank($submenu->getRank() + 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du sous-menu a été avancé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du sous-menu n\'a pas pu être avancé');
        }
        
        return $this->redirectToRoute('app_admin_submenu_index', [], Response::HTTP_SEE_OTHER);
    }
    
    #[Route('/down/{id}', name: 'app_admin_submenu_move_down', methods: ['POST'])]
    public function moveDown(Request $request, Submenu $submenu, EntityManagerInterface $em): Response
    {
        if ($submenu->getRank() > 0 && $this->isCsrfTokenValid('move-down'.$submenu->getId(), $request->getPayload()->getString('_token'))) {
            $submenu->setRank($submenu->getRank() - 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du sous-menu a été reculé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du sous-menu n\'a pas pu être reculé');
        }

        return $this->redirectToRoute('app_admin_submenu_index', [], Response::HTTP_SEE_OTHER);
    }
}
